;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Variable and Breed declarations ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

globals
[
  ;; system variables for program to be functional
  player-history    ;; the history of which number was the minority (encoded into a binary number), seen by the players
  android-history   ;; the history of which number was the minority (encoded into a binary number), seen by the androids
  minority          ;; the current number in the minority
  ticks             ;; keeps track of total run time of the game
  clock             ;; keeps track of the timer for players to make their decision
  avg-score         ;; keeps track of all turtles' average score
  stdev-score       ;; keeps track of the standard deviation of the turtles' scores
  avg-success       ;; keeps track of all turtles' average success

  winner
  loser

  ;; lists used to create the various turtles
  shape-names        ;; shapes available for players
  colors             ;; colors available for players
  color-names        ;; names of colors available for players
  used-shape-colors  ;; shape-color combinations used

  ;; quick start instructions and variables
  quick-start  ;; current quickstart instruction displayed in the quickstart monitor
  qs-item      ;; index of the current quickstart instruction
  qs-items     ;; list of quickstart instructions
]

turtles-own
[
  score          ;; each turtle's score
  choice         ;; each turtle's current choice
]

breeds
[
  androids
  players
]

players-own
[
  user-id        ;; user-id of hubnet player
  chosen-sides?  ;; true/false to tell if player has made current choice
  choices-made   ;; the number of choices each turtle has made
]

androids-own
[
  strategies        ;; each android's strategies (a list of lists)
  current-strategy  ;; each android's current strategy (index for the above list)
  strategies-scores ;; the accumulated virtual scores for each of the turtle's strategies
]


;;;;;;;;;;;;;;;;;;;;;
;; Setup Functions ;;
;;;;;;;;;;;;;;;;;;;;;

;; startup of the model
to startup
  setup
  setup-quick-start
  hubnet-set-client-interface "COMPUTER" [ "clients/Minority Game client.nlogo" ]
  hubnet-reset
end

;; setup for the overall program, will require clients to re-login
to setup
  ;; prevent an infinite loop from occurring in assign-strategies
  if (android-memory = 1 and strategies-per-android > 4 )
    [ user-message "You need to increase the memory variable or\n" +
                 "decrease the strategies-per-agent variable"
      stop ]
  ca
  setup-quick-start
  initialize-system
  initialize-androids
end

;; initializes system variables
to initialize-system
  clear-output
  ifelse (player-memory >= android-memory)
    [ set player-history random (2 ^ player-memory)
      set android-history full-history player-history player-memory
      while [ length android-history > android-memory ]
        [ set android-history but-first android-history ]
      set android-history (decimal android-history) ]
    [ set android-history random (2 ^ android-memory)
      set player-history full-history android-history android-memory
      while [ length player-history > player-memory ]
        [ set player-history but-first player-history ]
      set player-history (decimal player-history) ]
  set ticks 0
  set clock wait-time + 1
  set avg-score 0
  set stdev-score 0
  set winner "N/A"
  set loser "N/A"
  set shape-names [ "ant" "bee" "boat" "box" "butterfly" "circle" "person" "spacecraft"
                   "arrow 2" "truck" "turtle" "wolf" ]
  set colors      [ white   gray   brown   yellow   lime   turquoise
                    cyan   sky   violet   orange   magenta   pink ]
  set color-names [ "white" "gray" "brown" "yellow"  "lime" "turquoise"
                   "cyan" "sky" "violet" "orange" "magenta" "pink" ]
  set used-shape-colors []
end

;; reports the history in binary format (with padding if needed)
to-report full-history [ agent-history agent-memory ]
  let full binary agent-history
  while [length full < agent-memory]
    [ set full fput 0 full ]
  report full
end

;; converts a decimal number to a binary number (stored in a list of 0's and 1's)
to-report binary [ decimal-num ]
  let binary-num []
  loop
    [ set binary-num fput (decimal-num mod 2) binary-num
      set decimal-num int (decimal-num / 2)
      if (decimal-num = 0)
        [ report binary-num ] ]
end

;; converts a binary number (stored in a list of 0's and 1's) to a decimal number
to-report decimal [ binary-num ]
  report reduce [(2 * ?1) + ?2] binary-num
end

;; remove existing turtles and create number-of-androids androids
to initialize-androids
  ask androids
    [ die ]
  create-custom-androids number-of-androids
    [ set color black
      setxy 0 (screen-size-y * who / (count turtles))  ;; disperse over the y-axis
      set heading 90
      assign-strategies
      set current-strategy random strategies-per-android
      set choice item android-history (item current-strategy strategies)
      ifelse (color-by = "choice")
        [ recolor ]
        [ set color green ]  ;; we initially set all to green to prevent divide by zero error
      set score 0
      set strategies-scores n-values strategies-per-android [0] ]
  let num-picked-zero count turtles with [choice = 0]
  ifelse (num-picked-zero <= (count turtles - 1) / 2)
    [ set minority 0 ]
    [ set minority 1 ]
  setup-plots
end

;; gives the androids their allotted number of unique strategies
to assign-strategies  ;; android procedure
  let temp-strategy false
  set strategies []
  repeat strategies-per-android
    [ set temp-strategy create-strategy
      while [ member? temp-strategy strategies ]
        [ set temp-strategy create-strategy ]
      set strategies fput temp-strategy strategies ]
end

;; creates a strategy (a binary number stored in a list of
;; length 2 ^ android-memory)
to-report create-strategy
  report n-values (2 ^ android-memory) [random 2]
end

;; color the androids based on the color-by option
to recolor  ;; android procedure
  ifelse (color-by = "choice")
    [ recolor-by-choice ]
    [ recolor-by-speed ]
end

;; colors the androids according to their choice
to recolor-by-choice  ;; android procedure
  ifelse (choice = 0)
    [ set color red ]
    [ set color blue ]
end

;; colors the androids according to their success
to recolor-by-speed  ;; android procedure
  ifelse (score > avg-score + stdev-score)
    [ set color red ]
    [ ifelse (score < avg-score - stdev-score)
        [ set color blue ]
        [ set color green ] ]
end

;; resets the system, activated by clicking "re-run" button
to clear-all-data
  initialize-system
  initialize-androids
  ask players
    [ clear-my-data ]
  setup-plots
end

;; reset a player to some initial values
to clear-my-data  ;; players procedure
  ht
  setxy 0 (random-float screen-edge-y - random-float screen-edge-y)
  set choice (random 2)
  set score 0
  set chosen-sides? false
  set choices-made 0
  send-identity-to-clients
  clear-my-client
  st
end

;; allow HubNet clients to log in without running the model
to log-in
  every 0.1
    [ listen-clients ]
end


;;;;;;;;;;;;;;;;;;;;;;;
;; Runtime Functions ;;
;;;;;;;;;;;;;;;;;;;;;;;

to go
  ;; we need at least two turtles to do this experiment and to ensure that there is a minority
  ;; we increase this number to three.
  if count turtles < 3
    [ user-message "This requires at least three (3) turtles to run.  Please create more"
        + " androids or have more people log in."
      stop ]
  no-display  ;; turn off the display temporarily to speed up the model
  every 0.1
    [ ;; get commands and data from the clients
      listen-clients

      ;; determine if the system should be updated (advanced in time)
      if any? turtles
        [ if any? androids
            [ check-if-odd ]
          if clock = 0
            [ decide-for-them ]
          if all-players-decided?
            [ update-system
              update-scores-and-strategies
              advance-system
              update-choices
              do-plot
              if any? players
                [ set winner user-id-of random-one-of players with [ score = max values-from players [score] ]
                  set loser user-id-of random-one-of players with [ score = min values-from players [score] ] ]
              move
              show-extremes ] ] ]

  ;; controls the timer for clients to make a decision
  every 1
    [ set clock clock - 1
      hubnet-broadcast "Time to choose:" clock ]
  display
end

;; make sure that there will be a true minority
to check-if-odd
  if (count turtles mod 2 = 0)
    [ ask random-one-of androids
        [ die ] ]
end

;; if the time limit is up, make a random choice for clients
to decide-for-them
  ask players with [not chosen-sides?]
    [ set choice random 2
      set chosen-sides? true
      set choices-made choices-made + 1
      hubnet-send user-id "Current choice:" choice
      hubnet-send user-id "Status:" "Your selection was made for you." ]
end

;; reports if all the players have made a decision
to-report all-players-decided?
  report not any? players with [ not chosen-sides? ]
end

;; updates system variables such as minority, avg-score, and stdev-score globals
to update-system
  let num-picked-zero count turtles with [choice = 0]
  ifelse (num-picked-zero <= (count turtles - 1) / 2.0)
    [ set minority 0 ]
    [ set minority 1 ]
  print "Number Picked Zero: " + num-picked-zero + "\tMinority Choice: " + minority
  ;; plot this here (instead of in do-plot) for speed
  set-current-plot "Number Picking Zero"
  plot num-picked-zero

  set avg-score mean (values-from androids [score] + values-from players [score])
  set stdev-score standard-deviation (values-from androids [score] + values-from players [score])
  if ticks > 0
    [ set avg-success mean (values-from androids [score / ticks] + values-from players with [ choices-made > 0 ] [score / choices-made]) ]

  set clock wait-time + 1
end

;; ask all participants to update their strategy and scores
to update-scores-and-strategies
  ask androids
    [ update-androids-scores-and-strategies ]
  ask players
    [ update-score ]
end

;; if the turtle is in the minority, increase its score
to update-score  ;; turtle procedure
  if choice = minority
    [ set score score + 1 ]
end

;; updates android's score and their strategies' virtual scores
to update-androids-scores-and-strategies  ;; androids procedure
  let max-score last sort strategies-scores
  ;; this increases the virtual scores of the strategies if they selected the minority
  set strategies-scores (map
    [ ifelse-value (item android-history ?1 = minority)
        [?2 + 1]
        [?2] ]
    strategies strategies-scores)
  set max-score max strategies-scores
  let max-strategies []
  let counter 0
  ;; this picks a strategy with the largest virtual score
  foreach strategies-scores
    [ if (? = max-score)
        [ set max-strategies lput counter max-strategies ]
      set counter counter + 1 ]
  set current-strategy random-one-of max-strategies
  update-score
end

;; advances the system forward in time and updates the history
to advance-system
  set player-history decimal (lput minority but-first full-history player-history player-memory)
  set android-history decimal (lput minority but-first full-history android-history android-memory)
  ask players
    [ hubnet-send user-id "The past history:" full-history player-history player-memory ]
  set ticks ticks + 1
end

;; ask all participants to update their choice
to update-choices
  update-androids-choices
  update-players-choices
end

;; ask the androids to pick a new choice and then recolor themselves
to update-androids-choices
  ask androids
    [ set choice (item android-history (item current-strategy strategies))
      recolor ]
end

;; update the monitors on the client
to update-players-choices
  ask players
    [ hubnet-send user-id "Total score:" score
      hubnet-send user-id "Success rate:" round (score / choices-made * 1000) / 1000
      hubnet-send user-id "Your last choice:" choice
      hubnet-send user-id "Current choice:" "N/A"
      hubnet-send user-id "Status:" "Please make your selection."
      set chosen-sides? false ]
end

;; move turtles according to their success (a visual aid to see their collective behavior)
to move
  if avg-success > 0 and ticks > 0
    [ ask androids
        [ fd (score / ticks) / avg-success ]
      ask players
        [ fd (score / choices-made) / avg-success ] ]
end

;; label androids according to who is performing best and worst
to show-extremes
  let max-success max (values-from androids [score / ticks] + values-from players with [ choices-made > 0 ] [score / choices-made])
  let min-success min (values-from androids [score / ticks] + values-from players with [ choices-made > 0 ] [score / choices-made])
  ask androids
    [ set label no-label ]
  ask androids with [score / ticks = max-success]
    [ set label "best" ]
  ask androids with [score / ticks = min-success]
    [ set label "worst" ]
end


;;;;;;;;;;;;;;;;;;;
;; Plotting Code ;;
;;;;;;;;;;;;;;;;;;;

;; sets up plots and axis maxima
to setup-plots
  clear-all-plots
  set-current-plot "Number Picking Zero"
  ifelse any? turtles
    [ set-plot-y-range 0 (count turtles) ]
    [ set-plot-y-range 0 1 ]
  set-current-plot "Success Rates"
  set-histogram-num-bars 25
  do-plot  ;; plot initial value
end

;; updates three of the four plots.  one is updated in the update-system procedure for speed
to do-plot
  let score-list values-from turtles [score]
  let success-list values-from players with [ choices-made > 0 ] [score / choices-made]
  if ticks > 0
    [ set success-list success-list + (values-from androids [score / ticks ]) ]

  if length score-list > 0
    [ set-current-plot "Scores"
      set-current-plot-pen "max"
      plot max score-list
      set-current-plot-pen "min"
      plot min score-list
      set-current-plot-pen "avg"
      plot mean score-list ]

  if length success-list > 0
    [ set-current-plot "Success rate"
      set-current-plot-pen "max"
      plot max success-list
      set-current-plot-pen "min"
      plot min success-list
      set-current-plot-pen "avg"
      plot mean success-list

      set-current-plot "Success Rates"
      histogram-list success-list ]
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Quick Start functions ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; instructions to quickly setup the model, and clients to run this activity
to setup-quick-start
  set qs-item 0
  set qs-items
  [
    "Teacher: Follow these directions to run the HubNet activity."
    "Optional: Zoom In (see Tools in the Menu Bar)"
    "Optional: Change any of the settings...."
      "If you did change settings, press the SETUP button."
    "Teacher: Press the LOG-IN button."
    "Everyone: Open up a HubNet Client on your machine and..."
      "choose a user-name and..."
        "connect to this activity."
    "Teacher: Once everyone has started their client..."
      "press the LOG-IN button, then press GO."
    "Everyone: Watch your clock and choose 0 or 1."

    "Teacher: To rerun the activity with the same group,..."
      "stop the model by pressing the GO button, if it is on."
        "Change any of the settings that you would like."
          "Press the RE-RUN button."
    "Teacher: Restart the simulation by pressing the GO button again."

    "Teacher: To start the simulation over with a new group,..."
      "stop the model by pressing the GO button, if it is on..."
        "and follow these instructions again from the beginning."
  ]
  set quick-start (item qs-item qs-items)
end

;; view the next item in the quickstart monitor
to view-next
  set qs-item qs-item + 1
  if qs-item >= length qs-items
  [ set qs-item length qs-items - 1 ]
  set quick-start (item qs-item qs-items)
end

;; view the previous item in the quickstart monitor
to view-previous
  set qs-item qs-item - 1
  if qs-item < 0
  [ set qs-item 0 ]
  set quick-start (item qs-item qs-items)
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Code for interacting with the clients ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; listen for hubnet client activity
to listen-clients
  while [hubnet-message-waiting?]
  [ hubnet-fetch-message
    ifelse hubnet-enter-message?
      [ execute-create ]
      [ ifelse hubnet-exit-message?
          [ ask players with [user-id = hubnet-message-source]
              [ die ] ]
          [ if hubnet-message-tag = "0"
              [ choose-value 0 ]
            if hubnet-message-tag = "1"
              [ choose-value 1 ] ] ] ]
end

;; create a client player upon login
to execute-create
  create-custom-players 1
    [ set heading 90
      set user-id hubnet-message-source
      set size 3
      set-unique-shape-and-color
      clear-my-data ]
end

to set-unique-shape-and-color  ;; turtle procedure
  let max-possible-codes (length colors * length shape-names)
  let code random max-possible-codes
  while [member? code used-shape-colors and count turtles < max-possible-codes]
    [ set code random max-possible-codes ]
  set used-shape-colors (lput code used-shape-colors)
  set shape item (code mod length shape-names) shape-names
  set color item (code / length shape-names) colors
end

;; to tell the clients what they look like
to send-identity-to-clients  ;; player procedure
  hubnet-send user-id "You are a:" identity
end

;; report the string version of the turtle's identity (color + shape)
to-report identity  ;; turtle procedure
  report (color-string color) + " " + shape
end

;; report the string version of the turtle's color
to-report color-string [color-value]
  report item (position color-value colors) color-names
end

;; turtle/client procedure to reset the client interface
to clear-my-client  ;; player procedure
  hubnet-send user-id "Status:" "Please make your selection."
  hubnet-send user-id "Your last choice:" "N/A"
  hubnet-send user-id "The past history:" full-history player-history player-memory
  hubnet-send user-id "Total score:" "0"
  hubnet-send user-id "Success rate:" "N/A"
end

;; the client chooses 0 or 1
to choose-value [ value-chosen ]
  ask players with [user-id = hubnet-message-source]
    [ if (not chosen-sides?)
        [ set choice value-chosen
          set chosen-sides? true
          set choices-made choices-made + 1
          hubnet-send user-id "Current choice:" value-chosen
          hubnet-send user-id "Status:" "Waiting for other players..." ] ]
end


; *** NetLogo Model Copyright Notice ***
;
; This activity and associated models and materials were created as part of the projects:
; PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS and 
; INTEGRATED SIMULATION AND MODELING ENVIRONMENT.
; These projects gratefully acknowledge the support of the 
; National Science Foundation (REPP & ROLE programs) -- grant numbers 
; REC #9814682 and REC-0126227.
;
; Copyright 2004 by Uri Wilensky. Updated 2004. All rights reserved.
;
; Permission to use, modify or redistribute this model is hereby granted,
; provided that both of the following requirements are followed:
; a) this copyright notice is included.
; b) this model will not be redistributed for profit without permission
;    from the copyright holders.
; Contact the copyright holders for appropriate licenses for redistribution 
; for profit.
;
; To refer to this model in academic publications, please use:
; Wilensky, U. (2004).  NetLogo HubNet Minority Game model.
; http://ccl.northwestern.edu/netlogo/models/HubNetMinorityGame.
; Center for Connected Learning and Computer-Based Modeling,
; Northwestern University, Evanston, IL.
;
; In other publications, please use:
; Copyright 2004 Uri Wilensky.  All rights reserved.
; See http://ccl.northwestern.edu/netlogo/models/HubNetMinorityGame
; for terms of use.
;
; *** End of NetLogo Model Copyright Notice ***
@#$#@#$#@
GRAPHICS-WINDOW
415
96
803
505
17
17
10.8
1
10
1
1
1
0

CC-WINDOW
5
599
812
694
Command Center

BUTTON
36
10
105
43
setup
setup
NIL
1
T
OBSERVER
NIL
NIL

BUTTON
314
10
377
43
go
go
T
1
T
OBSERVER
NIL
NIL

SLIDER
20
87
198
120
number-of-androids
number-of-androids
0
1001
55
1
1
NIL

SLIDER
20
51
198
84
player-memory
player-memory
1
12
6
1
1
NIL

SLIDER
20
123
198
156
android-memory
android-memory
1
12
3
1
1
NIL

PLOT
224
435
404
585
Success rate
time
success rate
0.0
1.0
0.0
0.05
true
true
PENS
"max" 1.0 0 -65536 true
"min" 1.0 0 -16776961 true
"avg" 1.0 0 -11352576 true

PLOT
13
435
219
585
Number Picking Zero
time
number
0.0
1.0
0.0
25.0
true
false
PENS
"number" 1.0 0 -11352576 true

MONITOR
217
56
390
105
History
full-history player-history player-memory
3
1

PLOT
13
283
219
433
Scores
time
score
0.0
1.0
0.0
1.0
true
true
PENS
"max" 1.0 0 -65536 true
"min" 1.0 0 -16776961 true
"avg" 1.0 0 -11352576 true

CHOOSER
255
112
347
157
color-by
color-by
"choice" "success"
1

PLOT
224
283
404
433
Success Rates
score
number
0.0
1.0
0.0
10.0
true
true
PENS
"default" 0.02 1 -16777216 false

BUTTON
113
10
188
43
re-run
clear-all-data
NIL
1
T
OBSERVER
T
NIL

SLIDER
210
176
302
209
wait-time
wait-time
0
15
4
1
1
NIL

SLIDER
20
159
199
192
strategies-per-android
strategies-per-android
1
15
5
1
1
NIL

MONITOR
415
10
803
59
Quick Start Instructions-More in Info Window
quick-start
3
1

BUTTON
415
61
563
94
Reset Instructions
setup-quick-start
NIL
1
T
OBSERVER
T
NIL

BUTTON
720
61
803
94
NEXT>>>
view-next
NIL
1
T
OBSERVER
T
NIL

BUTTON
635
61
719
94
<<<PREV
view-previous
NIL
1
T
OBSERVER
T
NIL

MONITOR
19
223
190
272
Winner
winner
3
1

MONITOR
194
223
365
272
Loser
loser
3
1

BUTTON
230
10
301
43
log-in
log-in
T
1
T
OBSERVER
T
NIL

MONITOR
309
163
408
212
Time to Choose
clock
3
1

@#$#@#$#@
WHAT IS IT?
-----------
Minority Game is a simplified model of an economic market.  In each round agents choose to join one of two sides, 0 or 1. Those on the minority side at the end of a round earn a point.  This game is inspired by the "El Farol" bar problem.  

Each round, the live participants must choose 0 or 1. They can view their choice history for a specified number of previous turns, and may employ a finite set of strategies to make their decision. The record available to them shows which side, 0 or 1, was in the minority in previous rounds.

This HubNet version of the model allows players to play against each other and a set of androids.  The androids' intelligence (and thus the difficulty of the game) can be increased through the ANDROID-MEMORY slider.


HOW IT WORKS
------------
Each player begins with a score of 0 and must choose a side, 0 or 1, during each round. If their choice is in the minority they earn a point.  If they do not make their decision in WAIT-TIME seconds, they are randomly assigned to a side.

Each computer agent begins with a score of 0 and STRATEGIES-PER-AGENT strategies. Each of these strategies is a string of 0 and 1 choices, such as [0 1 1 1 0 0 1] that together represent the agents' possible plan of action (first choose 0, next time choose 1, next time choose 1, etc.). Initially, they choose a random one of these strategies to use. If their current strategy correctly predicted whether 0 or 1 would be the minority, they add one point to their score.  Each strategy also earns virtual points according to whether it would have been correct or not.  From then on, the agents will use their strategy with the highest virtual point total to predict whether they should select 0 or 1. Thus, for each android, the "fittest" strategies survive.

This strategy consists of a list of 1's and 0's that is 2^ANDROID-MEMORY long.  The choice the computer agent then makes is based off of the history of past choices.  This history is also a list of 1's and 0's that is ANDROID-MEMORY long, but it is encoded into a binary number.  The binary number is then used as an index into the strategy list to determine the choice.

This means that if there are only computer agents and no human participants, once the number of computer agents, the number of strategies, and the length of the historical record are chosen, all parameters are fixed and the behavior of the system is of interest.


HOW TO USE IT
-------------
QUICKSTART INSTRUCTIONS:
------------------------
Teacher: Follow these directions to run the HubNet activity.
Optional: Zoom In (see Tools in the Menu Bar)
Optional: Change any of the settings.  If you did change settings, press the SETUP button.
Teacher: Press the LOG-IN button.

Everyone: Open up a HubNet Client on your machine and choose a user-name and connect to this activity.

Teacher: Once everyone has started their client press the LOG-IN button, then press GO.

Everyone: Watch your clock and choose 0 or 1.

Teacher: To rerun the activity with the same group, stop the model by pressing the GO button, if it is on.
Change any of the settings that you would like.
Press the RE-RUN button.
Teacher: Restart the simulation by pressing the GO button again.

Teacher: To start the simulation over with a new group, stop the model by pressing the GO button, if it is on and follow these instructions again from the beginning.

BUTTONS:
--------
SETUP: Resets the simulation according to the parameters set by the sliders.  This should only be pressed when starting out with a new group of users since all data is lost.
RE-RUN: setup the model again for collecting more data or running the model again with the same users connected.
GO: Starts and stops the model.
LOG-IN: allows users to log into the activity without running the model
NEXT >>> - shows the next quick start instruction
<<< PREVIOUS - shows the previous quick start instruction
RESET INSTRUCTIONS - shows the first quick start instruction

SLIDERS:
--------
PLAYER-MEMORY: The length of the history the players can view to help choose sides.
NUMBER-OF-ANDROIDS: Sets the number of computer agents to participate. If the number of players and androids is an even number, an android will be removed to guarantee each time step features a minority.
ANDROID-MEMORY: Sets the length of the history which the computer agents use to predict their behavior.  One gets most interesting between 3 and 12, though there is some interesting behavior at 1 and 2.  Note that when using an ANDROID-MEMORY of 1, the STRATEGIES-PER-AGENT needs to be 4 or less.
STRATEGIES-PER-AGENT: Sets the number of strategies each computer agent has in their toolbox.  Five is typically a good value.  However, this can be changed for investigative purposes using the slider, if desired.
WAIT-TIME: the number of seconds to wait before automatically making a choice for any players that haven't chosen yet. So, as the clock counts down, if the clock is at '7' but all agents have input their choices, the clock will not continue ticking down to 0 but the next rounds will begin immediately.

CHOICE:
-------
COLOR-BY: "Choice" represents the computer agents changing their colors depending on if they have chosen 0 (red) or 1 (blue).  "Success" represents the computer agents changing their color depending upon their success rate (the number of times they have been in the minority divided by the number of selections).  An agent is green if they are within one standard deviation above and below the mean success rate, red if they are more than one standard deviation above, and blue if they are more than one standard deviation below.

MONITORS:
---------
WINNER: This is the player with the most accumulated points. If two or more players are ties, WINNER will show a random one of them.
LOSER: This is the player with the least accumulated points. If two or more players are ties, LOSER will show a random one of them.
HISTORY: shows the most recent minority values.  The number of values shown is determined by the PLAYER-MEMORY slider.

PLOTS:
------
SCORES: displays the minimum, maximum, and average scores over time
SCORES HISTOGRAM: a histogram of the scores that players and androids have gotten
NUMBER PICKING ZERO: plots the number of players and androids that picked zero during the last round
SUCCESS RATE: displays the minimum, maximum, and average success rate over time

**CAUTION: If you are on a slow computer and the model is likewise slow, try turning off the shapes in the graphics window.  You could also turn off the graphics window as the plots show everything the graphics window does, just in a different representation.  Also, beware setting the ANDROID-MEMORY slider to higher values.  It scales exponentially (2^ANDROID-MEMORY), however this only has an affect when SETUP is run.  This means that for each increased unit of ANDROID-MEMORY, it takes twice as long for SETUP to run.

CLIENT INFORMATION
------------------
After logging in, the client interface will appear for the students, and the participants will see many monitors and two buttons.  The object is for the participant to choose the value, 0 or 1, that she or he believes will be in the minority.

BUTTONS:
--------
0: press this button if you wish to choose 0 for a particular round.
1: press this button if you wish to choose 1 for a particular round.

MONITORS:
---------
YOU ARE A: displays the shape and color of your turtle in the graphics window
TOTAL SCORE: displays how many times you have chosen a value that has been in the minority
SUCCESS RATE: the number of times you have been in the minority divided by the number of selections you have participated in.
YOUR LAST CHOICE: the value you chose in the last round
THE PAST HISTORY: the values that were in the minority in the most recent previous rounds
CURRENT CHOICE: the value that you have chosen for this current round
STATUS: a message indicating the status of the game.  for example, whether you need to choose a value, or whether a value has been chosen for you (if you run out of time).
TIME TO CHOOSE: the number of seconds that remain in the round.  if you do not make a choice before the time has run out, a value is randomly chosen for you.


THINGS TO NOTICE
----------------
There are two extremes possible for each turn: the size of the minority is 1 agent or (NUMBER-OF-AGENTS - 1)/2 agents (since NUMBER-OF-AGENTS is always odd).  The former would represent a "wasting of resources" while the latter represents a situation which is more "for the common good."  However, each agent acts in an inherently selfish manner, as they care only if they and they alone are in the minority.  Nevertheless, the latter situation is prevalent in the system without live players.  Does this represent unintended cooperation between agents, or merely coordination and well developed powers of prediction?

The agents in the graphics window move according to how successful they are relative to the mean success rate.  After running for about 100 time steps (at just about any parameter setting), how do the fastest and slowest agents compare?  What does this imply?

Playing against others, what strategies seem to be the most effective?  What would happen if you simply chose randomly?

Look at the plot "Success Rates." As the game runs, the success rates converge. Can you explain this? At the time, the graph lines in the plot "Scores" diverge. Why is that?


THINGS TO TRY
-------------
What strategy works to maximize your own score?

Would you perform better against only computer agents than against humans?

What strategy works better to try to reach social equity?


EXTENDING THE MODEL
-------------------
Maybe you could add computer agents with different strategies, or more dynamically evolutionary strategies.  Could you figure out a strategy that works the best against these computer agents?  You could code in multiple dynamic strategies that play against each other. Who would emerge victorious?


NETLOGO FEATURES
----------------
One feature which was instrumental to this program being feasible was the n-values primitive.  When setting up strategies for each computer agent, they are binary numbers (stored in lists) of 2^ANDROID-MEMORY values. If this was done by starting with an empty list and using fput 2^ANDROID-MEMORY times, for each agent and for each strategy, during setup you would need to use fput N*S*(2^ANDROID-MEMORY) times.  Using n-values sped this up by about 2 or 3 orders of magnitude.

The list primitives 'map' and 'reduce' were also used to simplify code.


RELATED MODELS
--------------
Prisoner's Dilemma (Any of them)
Altruism
Cooperation
El Farol
Restaurants
The visual metaphor of relative success as a running competition, including the variation in color, is used in the 3 Doors model.

CREDITS AND REFERENCES
----------------------
Original implementation: Daniel B. Stouffer, for the Center for Connected Learning and Computer-Based Modeling.

This model was based upon studies by Dr. Damien Challet et al.  Information can be found on the web at http://www.unifr.ch/econophysics/minority/

Challet, D. and Zhang, Y.-C. Emergence of Cooperation and Organization in an Evolutionary Game. Physica A 246, 407 (1997).

Zhang, Y.-C. Modeling Market Mechanism with Evolutionary Games. Europhys. News 29, 51 (1998).

This activity and associated models and materials were created as part of the projects: PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN CLASSROOMS and INTEGRATED SIMULATION AND MODELING ENVIRONMENT. These projects gratefully acknowledge the support of the National Science Foundation (REPP & ROLE programs) -- grant numbers REC #9814682 and REC-0126227.

Copyright 2004 by Uri Wilensky. Updated 2004. All rights reserved.

Permission to use, modify or redistribute this model is hereby granted, provided that both of the following requirements are followed:
a) this copyright notice is included.
b) this model will not be redistributed for profit without permission from the copyright holders.
Contact the copyright holders for appropriate licenses for redistribution for profit.

To refer to this model in academic publications, please use: Wilensky, U. (2004). NetLogo HubNet Minority Game model. http://ccl.northwestern.edu/netlogo/models/HubNetMinorityGame. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

In other publications, please use: Copyright 2004 Uri Wilensky. All rights reserved. See http://ccl.northwestern.edu/netlogo/models/HubNetMinorityGame for terms of use.
@#$#@#$#@
default
true
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7566196 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

ant
true
0
Polygon -7566196 true true 136 61 129 46 144 30 119 45 124 60 114 82 97 37 132 10 93 36 111 84 127 105 172 105 189 84 208 35 171 11 202 35 204 37 186 82 177 60 180 44 159 32 170 44 165 60
Polygon -7566196 true true 150 95 135 103 139 117 125 149 137 180 135 196 150 204 166 195 161 180 174 150 158 116 164 102
Polygon -7566196 true true 149 186 128 197 114 232 134 270 149 282 166 270 185 232 171 195 149 186
Polygon -7566196 true true 225 66 230 107 159 122 161 127 234 111 236 106
Polygon -7566196 true true 78 58 99 116 139 123 137 128 95 119
Polygon -7566196 true true 48 103 90 147 129 147 130 151 86 151
Polygon -7566196 true true 65 224 92 171 134 160 135 164 95 175
Polygon -7566196 true true 235 222 210 170 163 162 161 166 208 174
Polygon -7566196 true true 249 107 211 147 168 147 168 150 213 150

arrow
true
0
Polygon -7566196 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

arrow 2
true
0
Polygon -7566196 true true 150 0 0 150 120 150 120 293 180 293 180 150 300 150

bee
true
0
Polygon -256 true false 152 149 77 163 67 195 67 211 74 234 85 252 100 264 116 276 134 286 151 300 167 285 182 278 206 260 220 242 226 218 226 195 222 166
Polygon -16777216 true false 150 149 128 151 114 151 98 145 80 122 80 103 81 83 95 67 117 58 141 54 151 53 177 55 195 66 207 82 211 94 211 116 204 139 189 149 171 152
Polygon -7566196 true true 151 54 119 59 96 60 81 50 78 39 87 25 103 18 115 23 121 13 150 1 180 14 189 23 197 17 210 19 222 30 222 44 212 57 192 58
Polygon -16777216 true false 70 185 74 171 223 172 224 186
Polygon -16777216 true false 67 211 71 226 224 226 225 211 67 211
Polygon -16777216 true false 91 257 106 269 195 269 211 255
Line -1 false 144 100 70 87
Line -1 false 70 87 45 87
Line -1 false 45 86 26 97
Line -1 false 26 96 22 115
Line -1 false 22 115 25 130
Line -1 false 26 131 37 141
Line -1 false 37 141 55 144
Line -1 false 55 143 143 101
Line -1 false 141 100 227 138
Line -1 false 227 138 241 137
Line -1 false 241 137 249 129
Line -1 false 249 129 254 110
Line -1 false 253 108 248 97
Line -1 false 249 95 235 82
Line -1 false 235 82 144 100

boat
false
0
Polygon -1 true false 63 162 90 207 223 207 290 162
Rectangle -6524078 true false 150 32 157 162
Polygon -16776961 true false 150 34 131 49 145 47 147 48 149 49
Polygon -7566196 true true 158 37 172 45 188 59 202 79 217 109 220 130 218 147 204 156 158 156 161 142 170 123 170 102 169 88 165 62
Polygon -7566196 true true 149 66 142 78 139 96 141 111 146 139 148 147 110 147 113 131 118 106 126 71

box
true
0
Polygon -7566196 true true 45 255 255 255 255 45 45 45

bug
true
0
Circle -7566196 true true 96 182 108
Circle -7566196 true true 110 127 80
Circle -7566196 true true 110 75 80
Line -7566196 true 150 100 80 30
Line -7566196 true 150 100 220 30

butterfly
true
0
Polygon -16777216 true false 151 76 138 91 138 284 150 296 162 286 162 91
Polygon -7566196 true true 164 106 184 79 205 61 236 48 259 53 279 86 287 119 289 158 278 177 256 182 164 181
Polygon -7566196 true true 136 110 119 82 110 71 85 61 59 48 36 56 17 88 6 115 2 147 15 178 134 178
Polygon -7566196 true true 46 181 28 227 50 255 77 273 112 283 135 274 135 180
Polygon -7566196 true true 165 185 254 184 272 224 255 251 236 267 191 283 164 276
Line -7566196 true 167 47 159 82
Line -7566196 true 136 47 145 81
Circle -7566196 true true 165 45 8
Circle -7566196 true true 134 45 6
Circle -7566196 true true 133 44 7
Circle -7566196 true true 133 43 8

car
false
0
Polygon -7566196 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7566196 true true 47 195 58
Circle -7566196 true true 195 195 58

circle
false
0
Circle -7566196 true true 30 30 240

circle 2
false
0
Circle -7566196 true true 16 16 270
Circle -16777216 true false 46 46 210

cow
false
0
Polygon -7566196 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7566196 true true 73 210 86 251 62 249 48 208
Polygon -7566196 true true 25 114 16 195 9 204 23 213 25 200 39 123

face happy
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7566196 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7566196 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7566196 true true 60 15 75 300
Polygon -7566196 true true 90 150 270 90 90 30
Line -7566196 true 75 135 90 135
Line -7566196 true 75 45 90 45

flower
false
0
Polygon -11352576 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7566196 true true 85 132 38
Circle -7566196 true true 130 147 38
Circle -7566196 true true 192 85 38
Circle -7566196 true true 85 40 38
Circle -7566196 true true 177 40 38
Circle -7566196 true true 177 132 38
Circle -7566196 true true 70 85 38
Circle -7566196 true true 130 25 38
Circle -7566196 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -11352576 true false 189 233 219 188 249 173 279 188 234 218
Polygon -11352576 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7566196 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7566196 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7566196 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7566196 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7566196 true 150 0 150 300

pentagon
false
0
Polygon -7566196 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7566196 true true 110 5 80
Polygon -7566196 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7566196 true true 127 79 172 94
Polygon -7566196 true true 195 90 240 150 225 180 165 105
Polygon -7566196 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7566196 true true 135 90 165 300
Polygon -7566196 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7566196 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7566196 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7566196 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7566196 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7566196 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7566196 true true 135 90 120 45 150 15 180 45 165 90

spacecraft
true
0
Polygon -7566196 true true 150 0 180 135 255 255 225 240 150 180 75 240 45 255 120 135

square
false
0
Rectangle -7566196 true true 30 30 270 270

square 2
false
0
Rectangle -7566196 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7566196 true true 60 270 150 0 240 270 15 105 285 105
Polygon -7566196 true true 75 120 105 210 195 210 225 120 150 75

target
false
0
Circle -7566196 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7566196 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7566196 true true 120 120 60

tree
false
0
Circle -7566196 true true 118 3 94
Rectangle -6524078 true false 120 195 180 300
Circle -7566196 true true 65 21 108
Circle -7566196 true true 116 41 127
Circle -7566196 true true 45 90 120
Circle -7566196 true true 104 74 152

triangle
false
0
Polygon -7566196 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7566196 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Polygon -7566196 true true 180 135 75 135 75 210 225 210 225 165 195 165
Polygon -8716033 true false 210 210 195 225 180 210
Polygon -8716033 true false 120 210 105 225 90 210

turtle
true
0
Polygon -11352576 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -11352576 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -11352576 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -11352576 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -11352576 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7566196 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7566196 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7566196 true 150 285 150 15
Line -7566196 true 15 150 285 150
Circle -7566196 true true 120 120 60
Line -7566196 true 216 40 79 269
Line -7566196 true 40 84 269 221
Line -7566196 true 40 216 269 79
Line -7566196 true 84 40 221 269

wolf
false
3
Polygon -6524078 true true 170 127 200 93 231 93 237 103 262 103 261 113 253 119 231 119 215 143 213 160 208 173 189 187 169 190 154 190 126 180 106 171 72 171 73 126 122 126 144 123 159 123
Polygon -6524078 true true 201 99 214 69 215 99
Polygon -6524078 true true 207 98 223 71 220 101
Polygon -6524078 true true 184 172 189 234 203 238 203 246 187 247 180 239 171 180
Polygon -6524078 true true 197 174 204 220 218 224 219 234 201 232 195 225 179 179
Polygon -6524078 true true 78 167 95 187 95 208 79 220 92 234 98 235 100 249 81 246 76 241 61 212 65 195 52 170 45 150 44 128 55 121 69 121 81 135
Polygon -6524078 true true 48 143 58 141
Polygon -6524078 true true 46 136 68 137
Polygon -6524078 true true 45 129 35 142 37 159 53 192 47 210 62 238 80 237
Line -16777216 false 74 237 59 213
Line -16777216 false 59 213 59 212
Line -16777216 false 58 211 67 192
Polygon -6524078 true true 38 138 66 149
Polygon -6524078 true true 46 128 33 120 21 118 11 123 3 138 5 160 13 178 9 192 0 199 20 196 25 179 24 161 25 148 45 140
Polygon -6524078 true true 67 122 96 126 63 144

x
false
0
Polygon -7566196 true true 270 75 225 30 30 225 75 270
Polygon -7566196 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 2.1beta4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
